<?php

/**
 * @package         Joomla.Plugin
 * @subpackage      Workflow.publishing
 *
 * @copyright   (C) 2020 Open Source Matters, Inc. <https://www.joomla.org>
 * @license         GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\Swjprojects\Customserverscheme\Extension;

use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;
use Joomla\Component\SWJProjects\Administrator\Event\ServerschemeEvent;
use Joomla\Component\SWJProjects\Administrator\Serverscheme\ServerschemePlugin;
use Joomla\Component\SWJProjects\Site\Helper\RouteHelper;
use Joomla\Event\SubscriberInterface;
use Joomla\Filesystem\File;

use function defined;

// phpcs:disable PSR1.Files.SideEffects
defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Workflow Publishing Plugin
 *
 * @since  4.0.0
 */
final class Customserverscheme extends ServerschemePlugin implements SubscriberInterface
{
    /**
     * Load the language file on instantiation.
     *
     * @var    bool
     * @since  4.0.0
     */
    protected $autoloadLanguage = true;

    /**
     * @var string
     * @since 2.5.0
     */
    protected string $mimeType = 'application/json';

    /**
     * @var string
     * @since 2.5.0
     */
    protected string $charset = 'utf-8';

    /**
     * @var string
     * @since 2.5.0
     */
    protected string $name = 'Custom Sample JSON';

    /**
     * @var string
     * @since 2.5.0
     */
    protected string $type = 'customsample';

    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return  array
     *
     * @since   4.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onGetServerschemeList' => 'onGetServerschemeList',
        ];
    }


    /**
     * @param   ServerschemeEvent  $event
     *
     *
     * @since 1.0.0
     */
    public function onGetServerschemeList(ServerschemeEvent $event): void
    {
        $event->addResult([
            'type'  => $this->getType(),
            'name'  => $this->getName(),
            'class' => $this,
        ]);
    }

    public function renderOutput(array $data): mixed
    {
        switch ($this->getScheme()) {
            case 'updates':
                $output = $this->buildProjectUpdatesJson($data);
                break;
            case 'changelogs':
                $output = $this->buildChangelogsJson($data);
                break;
            case 'collection':
            default:
                $output = $this->buildCollectionJson($data);
                break;
        }

        return $output;
    }

    protected function buildProjectUpdatesJson(array $data)
    {
        $updates    = [];
        $site_root  = Uri::getInstance()->toString(['scheme', 'host', 'port']);
        $files_root = $this->config['filesPath']['versions'];
        foreach ($data as $item) {
            // Add to updates
            $update = [
                'name'           => $item->name,
                'description'    => $item->description,
                'element'        => $item->element,
                'type'           => $item->type,
                'client'         => $item->client,
                'version'        => $item->version,
                'infourl'        => [
                    'url'   => $site_root . $item->link,
                    'title' => $item->name,
                ],
                'downloads'      => [],
                'tag'            => $item->tag,
                'targetPlatform' => [
                    'name'    => 'Custom Platform',
                    'version' => 100500,
                ],

            ];
            if ($item->type == 'plugin') {
                $update['folder'] = $item->folder;
            }

            if ($item->file) {
                $update['downloads'][] = [
                    'downloadurl' => $site_root . $item->download,
                    'type'        => 'full',
                    'format'      => File::getExt($item->file),
                ];

                $file_path_from_root    = $files_root . '/' . $item->id . '/' . $item->file;
                $update['sha256']       = hash_file('sha256', $file_path_from_root);
                $update['sha384']       = hash_file('sha384', $file_path_from_root);
                $update['sha512']       = hash_file('sha512', $file_path_from_root);
                $update['changelogurl'] = $site_root . Route::_(RouteHelper::getJChangelogRoute(null, $item->element));
            }
            $updates[] = $update;
        }

        return json_encode($updates);
    }

    protected function buildChangelogsJson(array $data)
    {
        $changelogs = [];
        foreach ($data as $item) {
            $changelog = [
                'element' => $item->element,
                'type'    => $item->type,
                'version' => $item->version,
            ];

            foreach ($item->changelog->toObject() as $value) {
                $value_type = (empty($value->type)) ? 'info' : $value->type;
                if (!array_key_exists($value_type, $changelog)) {
                    $changelog[$value_type] = [];
                }
                $changelog[$value_type][] = [
                    'title'       => $value->title,
                    'description' => $value->description,
                ];
            }

            $changelogs[] = $changelog;
        }

        return json_encode($changelogs);
    }

    protected function buildCollectionJson(array $data): mixed
    {
        $extensionset = [];
        $site_root    = Uri::getInstance()->toString(['scheme', 'host', 'port']);
        // Add to extensionset
        foreach ($data as $item) {
            $extension = [
                'name'       => $item->title,
                'element'    => $item->element,
                'type'       => $item->type,
                'client'     => $item->client,
                'detailsurl' => $site_root . $item->link,
                'version'    => $item->version,

            ];
            if ($item->type == 'plugin') {
                $extension['folder'] = $item->folder;
            }
            $extensionset[] = $extension;
        }

        return json_encode($extensionset);
    }
}
